#include "DXEngine.h"
#include "2DMathLib.h"
#include "DDUtil.h"


/////////////////////////////////////////////////////////////////////////////////////////
/////CLASS DEFINITIONS///////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
class Sparky2D;
class CFrameRate;
class CImage;
class CBob;


typedef struct tagPARTICLE
{
	bool	state;		//Is the particle alive or dead?
	long	id;			//User defined particle type
	float	x, y;		//Particle's location
	float	xv, yv;		//Particle's vector
	float	speed;		//Particle's speed
	long	life;		//How many iterations the particle lives for
	long	age;		//Particle's current age
	DWORD	color;		//Particle's color
} PARTICLE;


class Sparky2D //My 2D particle engine
{
public:

	Sparky2D();
	Sparky2D(long num);
	~Sparky2D();

	int		AddSpark(long id, float x, float y, float xv, float yv,       //Fire a spark on a set vector
					 float speed, long life, DWORD color);
	int		AddSpark(long id, float x, float y, UCHAR angle,			  //Fire a spark on a vector derived 	
					 float speed, long life, DWORD color);				  //from an angle
			
	void	SetGravity(float force) {gravity_force = force;}
	void	SetWind(float force) {wind_force = force;}

	void	UpdateSparks(void);			//Update all the sparks
	void	UpdateSparks(long id);		//Update the sparks of type id
	void	UpdateSpark(long index);	//Update a spark at given index int the spark array

	void	KillSparks(void);		//Kill all the sparks
	void	KillSparks(long id);	//Kill all the sparks of type id
	void	KillSpark(long index);	//Kill a spark ar a given index in the spark array

	int		DrawSparks(DXEngine *dxengine);				//Draw all the sparks on the back buffer
	int		DrawSparks(DXEngine *dxengine, long id);	//Draw all the sparks of type id on the back buffer

	int		GetSpark(PARTICLE *spark, long index);	//WARNING: Gives you a direct access pointer to a chosen spark
	int		GetSparkList(PARTICLE *spark_array);	//WARNING: Gives you direct access to the entire spark list
	long	GetNumSparks(void);		//Tells you the number of instances in the spark list 
	long	GetNumActive(void);		//Returns how many active sparks there are
private:
	
	float		gravity_force;
	float		wind_force;

	long		num_sparks;		//Stores the max number of sparks that memory is allocated for
	long		num_active;		//Stores the number of sparks that are alive
	PARTICLE	*spark_list;	//Stores the sparks themselves

	int			ReAllocateMem(void);	//If we run out of memory allocate more
};


class CFrameRate
{
public:
	CFrameRate();

	void CountFrames(void);
	int GetRate(void);
	void PrintRate(int x, int y, DWORD color, DXEngine *engine);

private:

	int FrameRate;
	int FrameCount;
	int FrameCount0;
	DWORD FrameTime;
	DWORD FrameTime0;
};


class CImage
{
public:
	CImage();
	~CImage();
	
	int AttachDXEngine(DXEngine *engine);

	int LoadBitmap(char *name);
	int SetBltFx(DDBLTFX *bltfx, DWORD flags);
	int SetColorKey(DWORD low_col, DWORD high_col);
	int SetSrcRect(int left, int top, int right, int bottom);
	int SetScale(float xs, float ys);
	int GetDC(HDC *hdc);
	int ReleaseDC(HDC hdc);
	int Blt(int x, int y);
	
	inline int Width(void){return width;}
	inline int Height(void){return height;}
	inline int Bpp(void){return bpp;}
	inline int Size(void){return size;}

protected:
	DXEngine *dxengine;

	char file[100];

	int width;
	int height;
	int bpp;
	int size;

	float xscale, yscale;

	RECT src_rect;
	int blt_width;
	int blt_height;
	DWORD bltflags;
	DDBLTFX ddbltfx;
	DDSURFACEDESC2 ddsd;
	LPDIRECTDRAWSURFACE7 lpdds;
};


class CBob
{
	friend CBob;
public:
	CBob();
	~CBob();

	int AttachDXEngine(DXEngine *engine);			//IMPORTANT must be called after dx initialization
	int AttachImageList(CImage *images, int num_images);	//IMPORTANT attaches images

	void SetAnimPattern(int *list, int length);		//Sets the flipping order for bob images
	void SetAnimPeriod(long frames);				//Sets the number of frames between images

	void SetPosition(int x, int y);					//Sets the bob's position
	void SetVector(float xv, float yv); 			//Sets the bob's vector directly
	void SetVector(UCHAR angle);					//Calculates a vector from an angle
	void SetStartSpeed(float speed);				//The bob's starting speed
	void SetAccel(float value);						//Sets an overall acceleration for the bob
	void SetFriction(float coeff);					//Sets the coefficient of friction
	void SetGravity(float value);					//Sets the force of gravity
	void SetAge(long curr, long max);				//Sets how many frames the bob lives for (inf. if max == -1)
	void SetHealth(long curr, long max);			//Sets how much damage the bob can take in hitpoints
	void SetMass(float mass);						//Sets the mass of the bob

	void AdjustPosition(float cx, float cy);		//Moves the bob's position
	void AdjustVector(float xvc, float yvc);			//Moves the bob's vector by the specified vector
	void AdjustVector(UCHAR angle);					//Moves the bob's vector by the specified vector
	void AdjustAccel(float amt);					//Adjusts the bob's acceleration
	void AdjustFriction(float amt);					//Adjusts the coefficient of friction
	void AdjustGravity(float amt);					//Adjusts the pull of gravity
	void AdjustCurrAge(long amt);					//Changes the bob's age by amt
	void AdjustHealth(long amt);					//Adjusts the bob's health by amt
	
	void DestroyBob(void);							//Kills the bob

	int Update(void);								//Updated all bob information (Call in main loop)
	int DrawBob(void);								//Draws the bob

	void ShowBob(bool flag) {show_bob = flag;}		//Shows or hides the bob

	bool IsAlive(void) {return (is_dead ? 0 : 1);}	//Is the bob still alive
	bool IsDead(void) {return is_dead;}				//Is the bob dead

	long Age(void) {return curr_age;}				//How many frames old is the bob
	long MaxAge(void) {return max_age;}				//How many frames is the bobs entire life
	long Health(void) {return curr_health;}		//What is the bob's current health 
	long MaxHealth(void) {return max_health;}	//What is the bob's maximum possible health 

private:
	DXEngine *dxengine;

	int num_anims;
	int *anim_order;
	
	float x, y;
	float xv, yv;
	float speed;
	float accel;
	float coeff_fric;
	float grav_const;

	long curr_age;
	long max_age;

	long curr_health;
	long max_health;

	float mass;

	bool show_bob;
	bool is_dead;

	int frame_count;	//How many frames have passed 
	int anim_index;		//the current image_list index being drawn
	int anim_period;	//How many frames pass between anim_index increments

	int num_images;
	CImage *image_list;
};
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CLASS DEFINITIONS////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////SPARKY2D CLASS MEMBERS//////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
Sparky2D::Sparky2D()
{
	long index;

	gravity_force = 0;
	wind_force = 0;

	num_sparks = 500;
	num_active = 0;
	spark_list = new PARTICLE[num_sparks];
	
	for(index=0; index<num_sparks; index++) spark_list[index].state = false;
}


Sparky2D::Sparky2D(long num)
{
	long index;
	
	gravity_force = 0;
	wind_force = 0;
	
	num_sparks = num;
	num_active = 0;
	if(num_sparks < 1) num_sparks = 500;
	spark_list = new PARTICLE[num_sparks];

	for(index=0; index<num_sparks; index++) spark_list[index].state = false;
}


Sparky2D::~Sparky2D()
{
	num_sparks = 0;
	num_active = 0;
	if(spark_list != NULL)
	{
		delete[] spark_list;
		spark_list = NULL;
	}
}


int	Sparky2D::AddSpark(long id, float x, float y, float xv, float yv, float speed, long life, DWORD color)
{
	long index;
	for(index=0; index<num_sparks; index++)
	{
		if(!spark_list[index].state)
		{
			spark_list[index].state = true;
			spark_list[index].id = id;
			spark_list[index].x = x;
			spark_list[index].y = y;
			spark_list[index].xv = xv;
			spark_list[index].yv = yv;
			spark_list[index].speed = speed;
			spark_list[index].life = life;
			spark_list[index].age = 0;
			spark_list[index].color = color;
			return 1;
		}
	}
	
	index = num_sparks;
	if(!ReAllocateMem()) return 0;
	
	spark_list[index].state = true;
	spark_list[index].id = id;
	spark_list[index].x = x;
	spark_list[index].y = y;
	spark_list[index].xv = xv;
	spark_list[index].yv = yv;
	spark_list[index].speed = speed;
	spark_list[index].life = life;
	spark_list[index].age = 0;
	spark_list[index].color = color;
	
	return 1;
}


int Sparky2D::AddSpark(long id, float x, float y, UCHAR angle, float speed, long life, DWORD color)
{
	float xv = COS256(angle);
	float yv = SIN256(angle);
	return AddSpark(id, x, y, xv, yv, speed, life, color);
}


void Sparky2D::UpdateSparks()
{
	long index;
	for(index=0; index<num_sparks; index++)
	{
		if(spark_list[index].state)
		{
			spark_list[index].x += (spark_list[index].xv * spark_list[index].speed);
			spark_list[index].y += (spark_list[index].yv * spark_list[index].speed);
			
			spark_list[index].xv += wind_force;
			spark_list[index].yv += gravity_force;

			if(spark_list[index].life != -1) 
			{	
				spark_list[index].age++;
			
				if(spark_list[index].age >= spark_list[index].life)
				{
					spark_list[index].state = false;		
				}
			}
		}
	}
}	


void Sparky2D::UpdateSparks(long id)
{
	long index;
	for(index=0; index<num_sparks; index++)
	{
		if(spark_list[index].state && (spark_list[index].id == id))
		{
			spark_list[index].x += (spark_list[index].xv * spark_list[index].speed);
			spark_list[index].y += (spark_list[index].yv * spark_list[index].speed);
			
			spark_list[index].xv += wind_force;
			spark_list[index].yv += gravity_force;

			if(spark_list[index].life != -1) 
			{	
				spark_list[index].age++;
			
				if(spark_list[index].age >= spark_list[index].life)
				{
					spark_list[index].state = false;		
				}
			}
		}
	}
}	


void Sparky2D::UpdateSpark(long index)
{
	if(spark_list[index].state)
	{
		spark_list[index].x += (spark_list[index].xv * spark_list[index].speed);
		spark_list[index].y += (spark_list[index].yv * spark_list[index].speed);
		
		spark_list[index].xv += wind_force;
		spark_list[index].yv += gravity_force;

		if(spark_list[index].life != -1) 
		{	
			spark_list[index].age++;
		
			if(spark_list[index].age >= spark_list[index].life)
			{
				spark_list[index].state = false;		
			}
		}
	}
}	


void Sparky2D::KillSparks(void)
{
	long index;
	for(index=0; index<num_sparks; index++)
	{
		spark_list[index].state = false;
		spark_list[index].id = 0;
		spark_list[index].x = 0;
		spark_list[index].y = 0;
		spark_list[index].xv = 0;
		spark_list[index].yv = 0;
		spark_list[index].speed = 0;
		spark_list[index].life = 0;
		spark_list[index].age = 0;
		spark_list[index].color = 0;
	}
}


void Sparky2D::KillSparks(long id)
{
	long index;
	for(index=0; index<num_sparks; index++)
	{	
		if(spark_list[index].id == id)
		{
			spark_list[index].state = false;
			spark_list[index].id = 0;
			spark_list[index].x = 0;
			spark_list[index].y = 0;
			spark_list[index].xv = 0;
			spark_list[index].yv = 0;
			spark_list[index].speed = 0;
			spark_list[index].life = 0;
			spark_list[index].age = 0;
			spark_list[index].color = 0;
		}
	}
}


void Sparky2D::KillSpark(long index)
{
	spark_list[index].state = false;
	spark_list[index].id = 0;
	spark_list[index].x = 0;
	spark_list[index].y = 0;
	spark_list[index].xv = 0;
	spark_list[index].yv = 0;
	spark_list[index].speed = 0;
	spark_list[index].life = 0;
	spark_list[index].age = 0;
	spark_list[index].color = 0;
}


int Sparky2D::DrawSparks(DXEngine *dxengine)
{
	if(!dxengine) return 0;

	long index;
	for(index=0; index<num_sparks; index++)
	{
		if(spark_list[index].state)
		{
			dxengine->SetPixel((int)spark_list[index].x, (int)spark_list[index].y, spark_list[index].color);
		}
	}
	return 1;
}


int Sparky2D::DrawSparks(DXEngine *dxengine, long id)
{
	if(!dxengine) return 0;

	long index;
	for(index=0; index<num_sparks; index++)
	{
		if(spark_list[index].state && (spark_list[index].id == id))
		{
			dxengine->SetPixel((int)spark_list[index].x, (int)spark_list[index].y, spark_list[index].color);
		}
	}
	return 1;
}


int	Sparky2D::GetSpark(PARTICLE *spark, long index)
{
	if(index >= num_sparks) return 0;
	spark = &spark_list[index];
	return 1;
}


int	Sparky2D::GetSparkList(PARTICLE *spark_array)
{
	if(!spark_list)
	{
		spark_array = NULL;
		return 0;
	}

	spark_array = spark_list;
	return 1;
}


long Sparky2D::GetNumSparks(void)
{
	return num_sparks;
}


long Sparky2D::GetNumActive(void)
{
	long index;
	num_active = 0;
	for(index=0; index<num_sparks; index++)
	{
		if(spark_list[index].state) num_active++;
	}

	return num_active;
}


int Sparky2D::ReAllocateMem()
{
	long index;
	PARTICLE *tmp_list;
	tmp_list = new PARTICLE[num_sparks + 500];
	
	if(spark_list != NULL)
	{
		memcpy(tmp_list, spark_list, sizeof(PARTICLE) * num_sparks);
		delete[] spark_list;
	}
	
	for(index=num_sparks; index<num_sparks+500; index++)
	{
		tmp_list[index].state = false;	
	}

	num_sparks += 500;
	spark_list = tmp_list;
	tmp_list = NULL;
	
	return 1;
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF SPARKY2D CLASS MEMBERS///////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////CFRAMERATE CLASS MEMBERS////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
CFrameRate::CFrameRate()
{
	FrameRate = 0;
	FrameCount = 0;
	FrameCount0 = 0;
	FrameTime = 0;
	FrameTime0 = 0;
}


void CFrameRate::CountFrames(void)
{
	FrameCount++;
	FrameTime = GetTickCount();

	if (FrameTime - FrameTime0 > 1000)
	{
		FrameRate = (FrameCount - FrameCount0) * 1000 / (FrameTime - FrameTime0);
		FrameTime0 = FrameTime;
		FrameCount0 = FrameCount;
	}
}


int CFrameRate::GetRate(void)
{
	return FrameRate;
}


void CFrameRate::PrintRate(int x, int y, DWORD color, DXEngine *engine)
{
	char buffer[50];
	sprintf(buffer, "FPS: %d", FrameRate);
	engine->SetText(x, y, buffer, color);
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CFRAMERATE CLASS MEMBERS/////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////CIMAGE CLASS MEMBERS////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
CImage::CImage()
{
	dxengine = NULL;

	width = 0;
	height = 0;
	bpp = 0;
	size = 0;

	xscale = 1;
	yscale = 1;

	src_rect.left = 0;
	src_rect.top = 0;
	src_rect.right = 0;
	src_rect.bottom = 0;

	blt_width = 0;
	blt_height = 0;

	bltflags = DDBLT_WAIT;
	ZeroMemory(&ddbltfx, sizeof(ddbltfx));
	ddbltfx.dwSize = sizeof(ddbltfx);
	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	lpdds = NULL;
}


CImage::~CImage()
{
	if(lpdds != NULL)
	{
		lpdds->Release();
		lpdds = NULL;
	}

	dxengine = NULL;
}


int CImage::AttachDXEngine(DXEngine *engine)
{
	if(engine == NULL) return 0;
	dxengine = engine;
	return 1;
}


int CImage::LoadBitmap(char *name)
{
	if(lpdds != NULL)
	{
		lpdds->Release();
		lpdds = NULL;
	}

	lpdds = DDLoadBitmap(dxengine->lpdd, name, 0, 0);
	if(lpdds == NULL) return 0;

	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	lpdds->GetSurfaceDesc(&ddsd);   

	strcpy(file, name);
	width = ddsd.dwWidth;
	height = ddsd.dwHeight;
	bpp = ddsd.ddpfPixelFormat.dwRGBBitCount;
	size = (width * height * (bpp / 8));

	src_rect.left = 0;
	src_rect.top = 0;
	src_rect.right = width;
	src_rect.bottom = height;

	blt_width = width;
	blt_height = height;

	bltflags = DDBLT_WAIT;

    return 1;
}


int CImage::SetBltFx(DDBLTFX *bltfx, DWORD flags)
{
	if(bltfx == NULL) return 0;
	bltflags = flags;
	memcpy(&ddbltfx, bltfx, sizeof(DDBLTFX)); 
	return 1;
}


int CImage::SetColorKey(DWORD low_col, DWORD high_col)
{
	DDCOLORKEY ddck;
	if(low_col > high_col) SWAP(low_col, high_col);
	ddck.dwColorSpaceLowValue = low_col;
	ddck.dwColorSpaceHighValue = high_col;
	lpdds->SetColorKey(DDCKEY_SRCBLT, &ddck);
	
	if(!(bltflags & DDBLT_KEYSRC))
	{
		bltflags |= DDBLT_KEYSRC;
	}
	return 1;
}


int CImage::SetSrcRect(int left, int top, int right, int bottom)
{
	if(left < 0) left = 0;
	if(right > width) right = width;
	if(top < 0) top = 0;
	if(bottom > height) bottom = height;

	src_rect.left = left;
	src_rect.right = right;
	src_rect.top = top;
	src_rect.bottom = bottom;

	blt_width = ABS(src_rect.right - src_rect.left);
	blt_height = ABS(src_rect.bottom - src_rect.top);
	return 1;
}


int CImage::SetScale(float xs, float ys)
{
	xscale = xs;
	yscale = ys;
	return 1;
}


int CImage::GetDC(HDC *hdc)
{
	if(FAILED(lpdds->GetDC(hdc))) return 0;
	return 1;
}


int CImage::ReleaseDC(HDC hdc)
{
	if(FAILED(lpdds->ReleaseDC(hdc)))return 0;
	return 1;
}

int CImage::Blt(int x, int y)
{
	RECT dest_rect;
	if(lpdds->IsLost())
	{
		DDReLoadBitmap(lpdds, file);
	}

	dest_rect.left = x;
	dest_rect.top = y;
	dest_rect.right = x + int(float(blt_width) * xscale);
	dest_rect.bottom = y + int(float(blt_height) * yscale);

	if(FAILED(dxengine->lpddsback->Blt(&dest_rect, lpdds, &src_rect, bltflags, &ddbltfx)))
	{
		return 0;
	}

	return 1;
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CIMAGE CLASS MEMBERS/////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////CBOB CLASS MEMBERS//////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
CBob::CBob()
{
	dxengine = NULL;

	num_anims = 0;
	anim_order = NULL;
	
	x = 0, y = 0;
	xv = 0, yv = 0;
	speed = 1;
	accel = 0;
	coeff_fric = 0;
	grav_const = 0;

	curr_age = 0;
	max_age = 0;

	curr_health = 0;
	max_health = 0;

	mass = 0;

	show_bob = false;
	is_dead = true;

	frame_count = 0;
	anim_index = 0;
	anim_period = 0;
	num_images = 0;
	image_list = NULL;
}


CBob::~CBob()
{
	dxengine = NULL;

	num_anims = 0;
	anim_order = NULL;
	
	x = 0, y = 0;
	xv = 0, yv = 0;
	speed = 1;
	accel = 0;
	coeff_fric = 0;
	grav_const = 0;

	curr_age = 0;
	max_age = 0;

	curr_health = 0;
	max_health = 0;

	mass = 0;

	show_bob = false;
	is_dead = true;

	frame_count = 0;
	anim_index = 0;
	anim_period = 0;
	num_images = 0;
	image_list = NULL;
}


int CBob::AttachDXEngine(DXEngine *engine)			//IMPORTANT must be called after dx initialization
{
	if(engine == NULL) return 0;
	dxengine = engine;
	return 1;
}


int CBob::AttachImageList(CImage *images, int num_images) //IMPORTANT attaches images
{
	if((images == NULL) || (num_images < 1)) return 0;
	image_list = images;
	CBob::num_images = num_images;
	return 1;
}


void CBob::SetAnimPattern(int *list, int length)	//Sets the flipping order for bob images
{
	if((list == NULL) || (length < 1)) return;
	anim_order = list;
	num_anims = length;
}


void CBob::SetAnimPeriod(long frames)				//Sets the number of frames between images
{
	anim_period = frames;
}


void CBob::SetPosition(int x, int y)				//Sets the bob's position
{
	CBob::x = (float)x;
	CBob::y = (float)y;
}


void CBob::SetVector(float xv, float yv)			//Sets the bob's vector directly
{
	CBob::xv = xv;
	CBob::yv = yv;
}


void CBob::SetVector(UCHAR angle)					//Calculates a vector from an angle
{
	SetVector(COS256(angle), SIN256(angle));
}


void CBob::SetStartSpeed(float speed)				//The bob's starting speed
{
	CBob::speed = speed;
}


void CBob::SetAccel(float value)					//Sets an overall acceleration for the bob
{
	accel = value;
}


void CBob::SetFriction(float coeff)					//Sets the coefficient of friction
{
	coeff_fric = coeff;
}


void CBob::SetGravity(float newtons)				//Sets the force of gravity
{
	grav_const = newtons;
}


void CBob::SetAge(long curr, long max)				//How many frames the bob lives for (inf. if max == -1)
{
	curr_age = curr;
	max_age = max;
	is_dead = 0;
}


void CBob::SetHealth(long curr, long max)			//How much damage the bob can take in hitpoints
{
	curr_health = curr;
	max_health = max;
}


void CBob::SetMass(float mass)						//Sets the mass of the bob
{
	CBob::mass = mass;
}


void CBob::AdjustPosition(float cx, float cy)		//Moves the bob's position
{
	x += cx;
	y += cy;
}


void CBob::AdjustVector(float xvc, float yvc)			//Moves the bob's vector by the specified vector
{
	xv += xvc;
	yv += yvc;
}


void CBob::AdjustVector(UCHAR angle)				//Moves the bob's vector by the specified vector
{
	AdjustVector(COS256(angle), SIN256(angle));
}


void CBob::AdjustAccel(float amt)					//Adjusts the bob's acceleration
{
	accel += amt;
}


void CBob::AdjustFriction(float amt)				//Adjusts the coefficient of friction
{
	coeff_fric += amt;
}


void CBob::AdjustGravity(float amt)					//Adjusts the pull of gravity
{
	grav_const += amt;
}


void CBob::AdjustCurrAge(long amt)					//Changes the bob's age by amt
{
	curr_age += amt;
}


void CBob::AdjustHealth(long amt)					//Adjusts the bob's health by amt
{
	curr_health += amt;
}


void CBob::DestroyBob(void)							//Kills the bob
{
	x = 0, y = 0;
	xv = 0, yv = 0;
	speed = 0;
	accel = 0;

	curr_age = 0;
	max_age = 0;

	curr_health = 0;
	max_health = 0;

	mass = 0;

	show_bob = false;
	is_dead = true;
}


int CBob::Update(void)								//Updated all bob information (Call in main loop)
{
	if(is_dead) return 0;

	speed += accel;
	yv += grav_const;

	x += xv * speed;
	y += yv * speed;
	
	if(max_age != -1)
	{
		curr_age++;
		if(curr_age >= max_age) DestroyBob();
	}

	if(max_health != -1)
	{
		if(curr_health <= 0) DestroyBob();
	}

	frame_count++;
	if(frame_count > anim_period)
	{
		frame_count = 0;
		anim_index++;
		if(anim_index >= num_anims) anim_index = 0;
	}

	return 1;
}


int CBob::DrawBob(void)								//Draws the bob
{
	if(is_dead) return 0;

	int i = anim_order[anim_index];
	if((i < 0) || (i >= num_images)) return 0;

	image_list[i].Blt((int)x, (int)y);
	return 1;
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CBOB CLASS MEMBERS///////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////

