/*************************************************************************

  Scene file support.

  $Id: scnfile.cxx,v 1.21 2001/04/09 19:40:41 garland Exp $

 *************************************************************************/


#include "ray318.h"
#include <map>
#include <vector>
#include <string>
#include <gfx/script.h>

////////////////////////////////////////////////////////////////////////
//
// Overall scene representation and management
//

Scene scene;

void Scene::add_object(rayObject *obj)
{
    obj->material = current_material();

    scene.objects.push_back(obj);
}

Mat4& Camera::compute_transform()
{
    M_ws = Mat4::I();
    M_ws = M_ws * viewport_matrix(width, height);

    M_ws = M_ws * perspective_matrix(fovy, aspect, 0.0, 0.0);

    M_ws = M_ws * lookat_matrix(eye, at, up);

    invert(M_sw, M_ws);

    return M_ws;
}

////////////////////////////////////////////////////////////////////////
//
// Lights and materials
//

Material::Material()
{
    emit = 0.0;
    r_diff = 0.0;
    r_spec = 0.0;
    r_amb = 0.0;
    r_trans = 0.0;

    shininess = 0.0;
    ir = 1.0;

	texture = NULL;
}

typedef map<string, Material *> matmap_t;
static matmap_t materials;

static Material default_material;
static Material *current = &default_material;

Material *current_material() { return current; }

Material *create_material(const string name)
{
    Material *mat = lookup_material(name);

    if( !mat )
    {
	mat = new Material;
	materials[name] = mat;
    }

    current = mat;
    return mat;
}

Material *lookup_material(const string name)
{
    matmap_t::iterator i = materials.find(name);

    if( i==materials.end() )
	return NULL;
    else
	return materials[name];
}

////////////////////////////////////////////////////////////////////////
//
// Current transformation stack management
//

typedef vector<Mat4> matrix_stack_t;
static matrix_stack_t vxform(1, Mat4::I());

static void mload(const Mat4& M)
{
    Mat4& top = vxform.back();
    top =  M;
}

static void mmult(const Mat4& M)
{
    Mat4& top = vxform.back();
    top = top * M;
}

void Scene::transform(Vec3& v, double w)
{
    Mat4& M = vxform.back();
    Vec4 p = M * Vec4(v, w);

    v = proj(p);
}

////////////////////////////////////////////////////////////////////////
//
// Actual scripting command handlers
//

static int script_push(const CmdLine &cmd)
{
    vxform.push_back(vxform.back());
    return SCRIPT_OK;
}

static int script_pop(const CmdLine &cmd)
{
    if( vxform.size()<=1 )
    {
	cerr << "WARNING: Ignoring pop on empty stack." << endl;
	return SCRIPT_ERR_SYNTAX;
    }

    vxform.pop_back();
    return SCRIPT_OK;
}

static int script_identity(const CmdLine &cmd)
{
    mload(Mat4::I());
    return SCRIPT_OK;
}

static int script_matrix(const CmdLine &cmd)
{
    Vec4 row[4];
    int i;
	vector<double> ns;

	if(cmd.collect_as_numbers(ns) != 16) return SCRIPT_ERR_SYNTAX;

    for(i=0; i<4; i++) {
		int j = i*4;
		row[i] = Vec4(ns[j], ns[j+1], ns[j+2], ns[j+3]);
	}

    Mat4 M(row[0], row[1], row[2], row[3]);
    if( cmd.opname() == "mmult" )
	mmult(M);
    else
	mload(M);

    return SCRIPT_OK;
}

// usage: trans <dx> <dy> <dz>
static int script_trans(const CmdLine &cmd)
{
    Vec3 v;

    if(cmd.collect_as_numbers(v, 3)!=3) return SCRIPT_ERR_SYNTAX;
    mmult(translation_matrix(v));
    return SCRIPT_OK;
}

static int script_scale(const CmdLine &cmd)
{
    Vec3 v;

    if(cmd.collect_as_numbers(v, 3)!=3) return SCRIPT_ERR_SYNTAX;
    mmult(scaling_matrix(v));
    return SCRIPT_OK;
}

static int script_rot(const CmdLine &cmd)
{
    double theta;
    Vec3 axis;
	vector<double> ns;

	if(cmd.collect_as_numbers(ns) != 4) return SCRIPT_ERR_SYNTAX;

	theta = ns[0];
	axis = Vec3(ns[1], ns[2], ns[3]);

    unitize(axis);		// make sure axis is a unit vector
    mmult(rotation_matrix_deg(theta, axis));
    return SCRIPT_OK;
}

static int script_viewport(const CmdLine &cmd)
{
    int vp[2];

    if(cmd.collect_as_numbers(vp, 2)!=2) return SCRIPT_ERR_SYNTAX;
    scene.camera.width = vp[0];
    scene.camera.height = vp[1];
    return SCRIPT_OK;
}

// usage: persp fovy aspect
static int script_persp(const CmdLine &cmd)
{
    const double zmin = 0.0;
    const double zmax = 0.0;
    double args[2];

    if(cmd.collect_as_numbers(args, 2)!=2) return SCRIPT_ERR_SYNTAX;
    scene.camera.fovy = args[0];
    scene.camera.aspect = args[1];
    return SCRIPT_OK;
}

static int script_lookat(const CmdLine &cmd)
{
	vector<double> ns;

	if (cmd.collect_as_numbers(ns) != 9) return SCRIPT_ERR_SYNTAX;

	scene.camera.eye = Vec3(ns[0], ns[1], ns[2]);
	scene.camera.at = Vec3(ns[3], ns[4], ns[5]);
	scene.camera.up = Vec3(ns[6], ns[7], ns[8]);

    return SCRIPT_OK;
}

// usage: material <name>
static int script_material(const CmdLine &cmd)
{
    string matname = cmd.token_to_string(0);
	if (matname == "") return SCRIPT_ERR_SYNTAX;

    create_material(matname);
    return SCRIPT_OK;
}

// usage: emit <r> <g> <b>
// usage: diffrefl <r> <g> <b>
// usage: specrefl <r> <g> <b>
// usage: spectrans <r> <g> <b>
// usage: ambrefl <r> <g> <b>
// usage: shine <shininess>
// usage: refract <refractive index>

static int script_refl(const CmdLine &cmd)
{
    Material *mat = current_material();
    Vec3 r;

    if(cmd.collect_as_numbers(r, 3)<3)  return SCRIPT_ERR_SYNTAX;
	string name = cmd.opname();

    if(name == "diffrefl")       mat->r_diff = r;
    else if(name == "specrefl")  mat->r_spec = r;
    else if(name == "emit")      mat->emit = r;
    else if(name == "spectrans") mat->r_trans = r;
    else if(name == "ambrefl")   mat->r_amb = r;
    else return SCRIPT_ERR_SYNTAX;

    return SCRIPT_OK;
}

static int script_shine(const CmdLine &cmd)
{
    Material *mat = current_material();
    double s;

    if(cmd.collect_as_numbers(&s, 1)<1)  return SCRIPT_ERR_SYNTAX;
	string name = cmd.opname();

    if(name == "shine")         mat->shininess = s;
    else if(name == "refract")  mat->ir = s;
    else return SCRIPT_ERR_SYNTAX;

    return SCRIPT_OK;
}

static int script_texture(const CmdLine &cmd)
{
    Material *mat = current_material();

    string texname = cmd.token_to_string(0);
	if (texname == "") return SCRIPT_ERR_SYNTAX;

	ByteRaster *tex = read_image(texname.data());
    mat->texture = tex;
    
	return SCRIPT_OK;
}

// usage: pointlight <x> <y> <z>  <r> <g> <b>
static int script_pointlight(const CmdLine &cmd)
{
    Light L;
	vector<double> ns;

    if (cmd.collect_as_numbers(ns) != 6) return SCRIPT_ERR_SYNTAX;

	L.position = Vec3(ns[0], ns[1], ns[2]);
	L.rgb = Vec3(ns[3], ns[4], ns[5]);

    scene.transform_point(L.position);
    scene.lights.push_back(L);
    return SCRIPT_OK;
}

static int script_background(const CmdLine &cmd)
{
    rgbColor bkg;
    if(cmd.collect_as_numbers(bkg, 3)!=3) return SCRIPT_ERR_SYNTAX;
    scene.background_color = bkg;
    return SCRIPT_OK;
}


static int script_ambient(const CmdLine &cmd)
{
    rgbColor amb;
    if(cmd.collect_as_numbers(amb, 3)!=3) return SCRIPT_ERR_SYNTAX;
    scene.ambient_light = amb;
    return SCRIPT_OK;
}

static double radius_scale(double r)
{
    Vec3 d[3];
    Vec3 s;

    for(int i=0; i<3; i++)
    {
	d[i] = 0.0;
	d[i][i] = r;

	scene.transform_direction(d[i]);
	s[i] = norm(d[i]) / r;
    }

    if( FEQ(s[0], s[1]) && FEQ(s[1], s[2]) )
	return s[0];
    else
    {
	cerr << "WARNING: Can't apply non-uniform scale to spheres!" << endl;
	return 1.0;
    }
}

// usage: sphere <c_x> <c_y> <c_z> <r>
static int script_sphere(const CmdLine &cmd)
{
    Vec3 ctr;
    double r;
	vector<double> ns;

	if(cmd.collect_as_numbers(ns) != 4) return SCRIPT_ERR_SYNTAX;

	ctr = Vec3(ns[0], ns[1], ns[2]);
	r = ns[3];

    scene.transform_point(ctr);
    r *= radius_scale(r);

    raySphere *sphere = new raySphere(r, ctr);
    scene.add_object(sphere);

    return SCRIPT_OK;
}

static int script_poly(const CmdLine &cmd)
{
    vector<Vec3> verts;
    Vec3 v;
	vector<double> ns;

	if (cmd.collect_as_numbers(ns) < 3) return SCRIPT_ERR_SYNTAX;

	int i=0;
    while( i+2 < ns.size() )
    {
		v = Vec3(ns[i], ns[i+1], ns[i+2]);
		i+=3;
		scene.transform_point(v);
		verts.push_back(v);
    }

    rayPolygon *p = new rayPolygon(verts.size(), verts.begin());
    scene.add_object(p);

    return SCRIPT_OK;
}

// usage: quadric <a> <b> <c> <d> <e> <f> <g> <h> <i> <j>
static int script_quadric(const CmdLine &cmd)
{
	vector<double> ns;
	Mat4 m;
	
	if (cmd.collect_as_numbers(ns) < 10) return SCRIPT_ERR_SYNTAX;

	m(0,0)=ns[0]; m(0,1)=ns[1]; m(0,2)=ns[2]; m(0,3)=ns[3]; 
	m(1,0)=ns[1]; m(1,1)=ns[4]; m(1,2)=ns[5]; m(1,3)=ns[6]; 
	m(2,0)=ns[2]; m(2,1)=ns[5]; m(2,2)=ns[7]; m(2,3)=ns[8]; 
	m(3,0)=ns[3]; m(3,1)=ns[6]; m(3,2)=ns[8]; m(3,3)=ns[9]; 

    rayQuadric *q = new rayQuadric(m);
	scene.add_object(q);

    return SCRIPT_OK;
}

void read_scene_file(const char *filename)
{
    CmdEnv env;
	env.register_command("push", script_push);
	env.register_command("pop", script_pop);
	env.register_command("identity", script_identity);
	env.register_command("trans", script_trans);
	env.register_command("rot", script_rot);
	env.register_command("scale", script_scale);
	env.register_command("mmult", script_matrix);
	env.register_command("mload", script_matrix);

	env.register_command("persp", script_persp);
	env.register_command("lookat", script_lookat);
	env.register_command("viewport", script_viewport);

	env.register_command("material", script_material);
	env.register_command("emit", script_refl);
	env.register_command("diffrefl", script_refl);
	env.register_command("specrefl", script_refl);
	env.register_command("spectrans", script_refl);
	env.register_command("ambrefl", script_refl);
	env.register_command("shine", script_shine);
	env.register_command("refract", script_shine);
	env.register_command("texture", script_texture);

	env.register_command("pointlight", script_pointlight);
	env.register_command("background", script_background);
	env.register_command("ambient", script_ambient);

	env.register_command("sphere", script_sphere);
	env.register_command("poly", script_poly);
	env.register_command("quadric", script_quadric);

	script_do_file(filename, env);
}