
////DirectX.H//////////////////////////////////////////////////////////////////
////By: Benjamin P. Gottemoller											   ////
////Copyright (c) 2000													   ////
////Purpose: To simplify DirectX code into small understandable functions. ////
///////////////////////////////////////////////////////////////////////////////

////Setup Instructions/////////////////////////////////////////////////////////
//// 1) Follow these instructions in your code:
////	main_window_handle = hwnd;  //where hwnd is your window handle
////	main_instance = hInstance;  //where hInstance is your window handle instance
////
//// 2) Do NOT declare/use new versions of the following data types:
////	LPDIRECTDRAW lpdd4 = NULL;
////	LPDIRECTDRAWSURFACE lpddsprimary = NULL; //DD4
////	LPDIRECTDRAWSURFACE lpddsback = NULL; //DD4
////	DDSURFACEDESC ddsd; //DD2
////	LPDIRECTDRAWCLIPPER lpddclipper = NULL;
////	Use the ones that have already been declared in this header.
///////////////////////////////////////////////////////////////////////////////


#ifndef _DIRECTX_H_
#define _DIRECTX_H_

/////////////////////////////////////////////////////////////////////
#define WIN32_LEAN_AND_MEAN
#define INITGUID

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <objbase.h>
#include <iostream.h> 
#include <conio.h>
#include <stdlib.h>
#include <malloc.h>
#include <memory.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <math.h>
#include <io.h>
#include <fcntl.h>

#include <ddraw.h>    
/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
#define KEY_DOWN(vk_code) ((GetAsyncKeyState(vk_code) & 0x8000) ? 1 : 0)
#define KEY_UP(vk_code)   ((GetAsyncKeyState(vk_code) & 0x8000) ? 0 : 1)
#define BITMAP_ID 0x4D42
/////////////////////////////////////////////////////////////////////


#ifndef _USEFUL_MACROS
#define _USEFUL_MACROS
const double PI = 3.14159265358979323846264338327950288419716939937510f;
const double PI2 = PI * 2;
const double RADIAN = PI / 180; 

#define SIN360(theta) (float)sin((double)theta * RADIAN) 
#define COS360(theta) (float)cos((double)theta * RADIAN) 
#define TAN360(theta) (float)tan((double)theta * RADIAN)
#define SWAP(a,b) ((a)^=(b)^=(a)^=(b))
#define RGB16(r,g,b) (USHORT)((b%32) + ((g%64) << 6) + ((r%32) << 11))
#define RGB32(a,r,g,b) (UINT)((b) + ((g) << 8) + ((r) << 16) + ((a) << 24))
#endif


/////////////////////////////////////////////////////////////////////
int ScreenX;
int ScreenY;
int ScreenBPP;
/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
HWND main_window_handle = NULL; 
HINSTANCE main_instance = NULL;    
/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
LPDIRECTDRAW lpdd4 = NULL;
LPDIRECTDRAWSURFACE lpddsprimary = NULL; //DD4
LPDIRECTDRAWSURFACE lpddsback = NULL; //DD4
DDSURFACEDESC ddsd; //DD2
LPDIRECTDRAWCLIPPER lpddclipper = NULL;
/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
class PIC
{
public:
	PIC(){x=y=xv=yv=width=height=0;}
	~PIC(){lpddsImage->Release();}
	int x,y;            
	int xv,yv;          
	int width, height;  
	LPDIRECTDRAWSURFACE lpddsImage; 
};
/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
int DD_Init(int width, int height, int bpp);
int DD_Shutdown(void);
int DD_Flip(void);
int DD_Wait_Retrace(void);
int DD_LockPrimary(void);
int DD_UnlockPrimary(void);
int DD_LockBack(void);
int DD_UnlockBack(void);
int DD_Fill_Surface(LPDIRECTDRAWSURFACE lpdds, unsigned long color);
int DD_Fill_Rect(LPDIRECTDRAWSURFACE lpdds, int x, int y, int width, int height, unsigned long color);
LPDIRECTDRAWCLIPPER DD_Attach_Clipper(LPDIRECTDRAWSURFACE lpdds, int num_rects, LPRECT clip_list);
LPDIRECTDRAWSURFACE DD_Create_Surface(int width, int height, int mem_flags);

LPDIRECTDRAWSURFACE DDLoadBitmap(LPCSTR szBitmap, int dx, int dy);
HRESULT DDSurfaceLoadBitmap(LPDIRECTDRAWSURFACE lpdds, LPCSTR szBitmap);
int DDCopyBitmap(LPDIRECTDRAWSURFACE lpdds, HBITMAP hBmp, int x, int y, int dx, int dy);

int DD_BitmapToPic(PIC *pic, LPCSTR szBitmap);
int DD_BltPic(PIC *pic, LPDIRECTDRAWSURFACE lpdds);
int DD_TMap(PIC *pic, LPDIRECTDRAWSURFACE lpdds, int x, int y, int width, int height);

void PROCESS_ERROR(char *error_description);

/////////////////////////////////////////////////////////////////////


BOOL LockModePrimary = 0;
BOOL LockModeBack = 0;


int DD_Init(int width, int height, int bpp)
{
	LPRECT tmpClipList = new RECT;
	tmpClipList->left = 0;
	tmpClipList->top = 0;
	tmpClipList->right = width;
	tmpClipList->bottom = height;

	if(FAILED(DirectDrawCreate(NULL, &lpdd4, NULL)))
	{
		PROCESS_ERROR("DD_Init: DirectDrawCreate()");
		return 0;
	}
												//should be IID_DirectDraw4
	if(FAILED(lpdd4->QueryInterface(IID_IDirectDraw, (LPVOID *)&lpdd4)))
	{
		PROCESS_ERROR("DD_Init: QueryInterface()");
		return 0;
	}

	if(FAILED(lpdd4->SetCooperativeLevel(main_window_handle, DDSCL_FULLSCREEN | DDSCL_ALLOWMODEX |
										 DDSCL_EXCLUSIVE | DDSCL_ALLOWREBOOT)))
	{
		PROCESS_ERROR("DD_Init: SetCooperativeLevel()");
		return 0;
	}

	if(FAILED(lpdd4->SetDisplayMode(width, height, bpp)))
	{
		PROCESS_ERROR("DD_Init: SetDisplayMode()");
		return 0;
	}
	ScreenX = width;
	ScreenY = height;
	ScreenBPP = bpp;

	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
	
	ddsd.dwBackBufferCount = 1;

	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_COMPLEX | DDSCAPS_FLIP;

	if(FAILED(lpdd4->CreateSurface(&ddsd, &lpddsprimary, NULL)))
	{
		PROCESS_ERROR("DD_Init: CreateSurface()");
		return 0;
	}

	ddsd.ddsCaps.dwCaps = DDSCAPS_BACKBUFFER;

	if(FAILED(lpddsprimary->GetAttachedSurface(&ddsd.ddsCaps, &lpddsback)))
	{
		PROCESS_ERROR("DD_Init: GetAttachedSurface()");
		return 0;
	}

	if(!DD_Attach_Clipper(lpddsback, 1, tmpClipList))
	{
		PROCESS_ERROR("DD_Init: DD_Attach_Clipper()");
		return 0;
	}

	return 1;
}



int DD_Shutdown(void)
{
	if(lpddsback)
	{
		lpddsback->Release();
		lpddsback = NULL;
	}

	if(lpddsprimary)
	{
		lpddsprimary->Release();
		lpddsprimary = NULL;
	}

	if(lpdd4)
	{
		lpdd4->Release();
		lpdd4 = NULL;
	}

	return 1;
}


int DD_Flip(void)
{
	if(!LockModePrimary && !LockModeBack)
	{
		while(FAILED(lpddsprimary->Flip(NULL, DDFLIP_WAIT)));
		return(1);
	}
	else
	{
		return 0;
	}
}


int DD_Wait_Retrace(void)
{
	lpdd4->WaitForVerticalBlank(DDWAITVB_BLOCKBEGIN,0);
	return(1);
}


int DD_Fill_Surface(LPDIRECTDRAWSURFACE lpdds,unsigned long color)
{
	DDBLTFX ddbltfx; 

	memset(&ddbltfx, 0, sizeof(ddbltfx));
	ddbltfx.dwSize = sizeof(ddbltfx);

	ddbltfx.dwFillColor = color; 

	lpdds->Blt(NULL,
		       NULL,
			   NULL,
			   DDBLT_COLORFILL | DDBLT_WAIT,  
			   &ddbltfx);  

	return(1);
} 


int DD_Fill_Rect(LPDIRECTDRAWSURFACE lpdds, int x, int y, int width, int height, unsigned long color)
{
	DDBLTFX ddbltfx;
	RECT rect;
	memset(&ddbltfx, 0, sizeof(ddbltfx));
	ddbltfx.dwSize = sizeof(ddbltfx);

	ddbltfx.dwFillColor = color;

	rect.left = x;
	rect.top = y;
	rect.right = x + width;
	rect.bottom = y + height;

	lpdds->Blt(&rect, 
			   NULL, 
			   NULL, 
			   DDBLT_COLORFILL | DDBLT_WAIT, 
			   &ddbltfx);

	return 1;
}



int DD_LockPrimary(void)
{
	if(LockModePrimary == 0)
	{
		lpddsprimary->Lock(NULL, &ddsd, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR, NULL);
		LockModePrimary = 1;
		return 1;
	}
	else
	{
		return 0;
	}
}

int DD_UnlockPrimary(void)
{
	if(LockModePrimary == 1)
	{
		lpddsprimary->Unlock(NULL);
		LockModePrimary = 0;
		return 1;
	}
	else
	{
		return 0;
	}
}

int DD_LockBack(void)
{
	if(LockModeBack == 0)
	{
		lpddsback->Lock(NULL, &ddsd, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR, NULL);
		LockModeBack = 1;
		return 1;
	}
	else
	{
		return 0;
	}
}

int DD_UnlockBack(void)
{
	if(LockModeBack == 1)
	{
		lpddsback->Unlock(NULL);
		LockModeBack = 0;
		return 1;
	}
	else
	{
		return 0;
	}
}


LPDIRECTDRAWCLIPPER DD_Attach_Clipper(LPDIRECTDRAWSURFACE lpdds, int num_rects, LPRECT clip_list)
{
	int index = 0;
	LPDIRECTDRAWCLIPPER lpddclipper;
	LPRGNDATA region_data;

	if(FAILED(lpdd4->CreateClipper(0, &lpddclipper, NULL)))
	{
		PROCESS_ERROR("DD_Attach_Clipper: CreateClipper()");
		return NULL;
	}
	
	region_data = (LPRGNDATA)malloc(sizeof(RGNDATAHEADER) + num_rects*sizeof(RECT));
	memcpy(region_data->Buffer, clip_list, sizeof(RECT)*num_rects);

	region_data->rdh.dwSize			= sizeof(RGNDATAHEADER);
	region_data->rdh.iType			= RDH_RECTANGLES;
	region_data->rdh.nCount			= num_rects;
	region_data->rdh.nRgnSize		= num_rects*sizeof(RECT);

	region_data->rdh.rcBound.left	=  64000;
	region_data->rdh.rcBound.top	=  64000;
	region_data->rdh.rcBound.right	= -64000;
	region_data->rdh.rcBound.bottom	= -64000;

	for(index = 0; index < num_rects; index++)
	{
		if(clip_list[index].left < region_data->rdh.rcBound.left)
		{
			region_data->rdh.rcBound.left = clip_list[index].left;
		}

		if(clip_list[index].right > region_data->rdh.rcBound.right)
		{
			region_data->rdh.rcBound.right = clip_list[index].right;
		}

		if(clip_list[index].top < region_data->rdh.rcBound.top)
		{
			region_data->rdh.rcBound.top = clip_list[index].top;
		}

		if(clip_list[index].bottom > region_data->rdh.rcBound.bottom)
		{
			region_data->rdh.rcBound.bottom = clip_list[index].bottom;
		}
	}

	if(FAILED(lpddclipper->SetClipList(region_data, 0)))
	{
		free(region_data);
		PROCESS_ERROR("DD_Attach_Clipper: SetClipList()");
		return NULL;
	}

	if(FAILED(lpdds->SetClipper(lpddclipper)))
	{
		free(region_data);
		PROCESS_ERROR("DD_Attach_Clipper: SetClipper()");
		return NULL;
	}

	free(region_data);
	return(lpddclipper);
}



LPDIRECTDRAWSURFACE DD_Create_Surface(int width, int height, int mem_flags)
{
	DDSURFACEDESC ddsd;
	LPDIRECTDRAWSURFACE lpdds;

	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);

	ddsd.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
	ddsd.dwWidth = width;
	ddsd.dwHeight = height;
	ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | mem_flags;

	
	if(FAILED(lpdd4->CreateSurface(&ddsd, &lpdds, NULL)))
	{
		PROCESS_ERROR("DD_Create_Surface: CreateSurface()");
		return NULL;
	}

	DDCOLORKEY color_key;
	color_key.dwColorSpaceLowValue = 0;
	color_key.dwColorSpaceHighValue = 0;

	if(FAILED(lpdds->SetColorKey(DDCKEY_SRCBLT, &color_key)))
	{
		PROCESS_ERROR("DD_Create_Surface: SetColorKey()");
		return NULL;
	}

	return(lpdds);
}



LPDIRECTDRAWSURFACE DDLoadBitmap(LPCSTR szBitmap, int dx, int dy)
{
    HBITMAP                 hBmp;
    BITMAP                  bmp;
    DDSURFACEDESC          ddsd;
    LPDIRECTDRAWSURFACE     lpdds;

    hBmp = (HBITMAP) LoadImage(GetModuleHandle(NULL), szBitmap, IMAGE_BITMAP, dx, dy, LR_CREATEDIBSECTION);
    
	if (hBmp == NULL)
    {
		hBmp = (HBITMAP) LoadImage(NULL, szBitmap, IMAGE_BITMAP, dx, dy, LR_LOADFROMFILE | LR_CREATEDIBSECTION);
    }

	if (hBmp == NULL)
    {
		PROCESS_ERROR("DDLoadBitmap: Cannot find bitmap file.");
		return NULL;
    }

	GetObject(hBmp, sizeof(bmp), &bmp);
    
	memset(&ddsd, 0, sizeof(ddsd));
    
	ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
    ddsd.dwWidth = bmp.bmWidth;
    ddsd.dwHeight = bmp.bmHeight;
    
	if (lpdd4->CreateSurface(&ddsd, &lpdds, NULL) != DD_OK)
    {
		PROCESS_ERROR("DDLoadBitmap: CreateSurface()");
		return NULL;
    }
	DDCopyBitmap(lpdds, hBmp, 0, 0, 0, 0);
    DeleteObject(hBmp);
    return lpdds;
}



HRESULT DDSurfaceLoadBitmap(LPDIRECTDRAWSURFACE lpdds, LPCSTR szBitmap)
{
    HBITMAP                 hBmp;
    HRESULT                 hr;

    hBmp = (HBITMAP) LoadImage(GetModuleHandle(NULL), szBitmap, IMAGE_BITMAP, 0, 0, LR_CREATEDIBSECTION);
    if (hBmp == NULL)
	{
		hBmp = (HBITMAP) LoadImage(NULL, szBitmap, IMAGE_BITMAP, 0, 0, LR_LOADFROMFILE | LR_CREATEDIBSECTION);
    }
	
	if (hBmp == NULL)
    {
        PROCESS_ERROR("DDSurfaceLoadBitmap: Cannot find bitmap file.");
        return 0;
    }

    hr = DDCopyBitmap(lpdds, hBmp, 0, 0, 0, 0);
    if (hr != DD_OK)
    {
        PROCESS_ERROR("DDSurfaceLoadBitmap: DDCopyBitmap()");
		return 0;
    }

    DeleteObject(hBmp);
    return hr;
}


int DDCopyBitmap(LPDIRECTDRAWSURFACE lpdds, HBITMAP hBmp, int x, int y, int dx, int dy)
{
    HDC                     hdcImage;
    HDC                     hdc;
    BITMAP                  bmp;
    DDSURFACEDESC          ddsd;
   
    if (hBmp == NULL || lpdds == NULL)
    {
		return E_FAIL;
    }

	lpdds->Restore();
    
	hdcImage = CreateCompatibleDC(NULL);
    
	if (!hdcImage)
    {
		PROCESS_ERROR("DDCopyBitmap: CreateCompatibleDC()");
		return 0;
	}
	
	SelectObject(hdcImage, hBmp);
    
	GetObject(hBmp, sizeof(bmp), &bmp);
    
	dx = dx == 0 ? bmp.bmWidth : dx;     
    dy = dy == 0 ? bmp.bmHeight : dy;
    
	ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_HEIGHT | DDSD_WIDTH;
    lpdds->GetSurfaceDesc(&ddsd);

    if (!FAILED(lpdds->GetDC(&hdc)))
    {
        StretchBlt(hdc, 0, 0, ddsd.dwWidth, ddsd.dwHeight, hdcImage, x, y, dx, dy, SRCCOPY);
        lpdds->ReleaseDC(hdc);
    }
    DeleteDC(hdcImage);
    return 1;
}



int DD_BitmapToPic(PIC *pic, LPCSTR szBitmap)
{
	int dx = 0;
	int dy = 0;
    HBITMAP                 hBmp;
    BITMAP                  bmp;
  
    hBmp = (HBITMAP) LoadImage(GetModuleHandle(NULL), szBitmap, IMAGE_BITMAP, dx, dy, LR_CREATEDIBSECTION);
    
	if (hBmp == NULL)
    {
		hBmp = (HBITMAP) LoadImage(NULL, szBitmap, IMAGE_BITMAP, dx, dy, LR_LOADFROMFILE | LR_CREATEDIBSECTION);
    }

	if (hBmp == NULL)
    {
		PROCESS_ERROR("DDLoadBitmap: Cannot find bitmap file.");
		return NULL;
    }

	GetObject(hBmp, sizeof(bmp), &bmp);

	pic->width = bmp.bmWidth;
	pic->height = bmp.bmWidth;
	if(!(pic->lpddsImage = DD_Create_Surface(bmp.bmWidth, bmp.bmHeight, DDSCAPS_VIDEOMEMORY)))
	{
		PROCESS_ERROR("DD_BitmapToPic: DD_Create_Surface()");
		return 0;
	}

	if(!DDCopyBitmap(pic->lpddsImage, hBmp, 0, 0, 0, 0))
	{
		PROCESS_ERROR("DD_BitmapToPic: DDCopyBitmap()");
		return 0;
	}

    DeleteObject(hBmp);

	return 1;
}


int DD_BltPic(PIC *pic, LPDIRECTDRAWSURFACE lpdds)
{
	RECT source_rect;
	RECT dest_rect;
	
	dest_rect.left = pic->x;
	dest_rect.top = pic->y;
	dest_rect.right = pic->x + pic->width - 1;
	dest_rect.bottom = pic->y + pic->height - 1;

	source_rect.left = 0;
	source_rect.top = 0;
	source_rect.right = pic->width;
	source_rect.bottom = pic->height;

	if(FAILED(lpdds->Blt(&dest_rect, pic->lpddsImage, &source_rect, (DDBLT_WAIT | DDBLT_KEYSRC), NULL)))
	{
		PROCESS_ERROR("DD_BltPic: Blt()");
		return 0;
	}

	return 1;
}


int DD_TMap(PIC *pic, LPDIRECTDRAWSURFACE lpdds, int x, int y, int width, int height)
{
	RECT source_rect;
	RECT dest_rect;
	
	dest_rect.left = x;
	dest_rect.top = y;
	dest_rect.right = x+width-1;
	dest_rect.bottom = y+height-1;

	source_rect.left = 0;
	source_rect.top = 0;
	source_rect.right = pic->width;
	source_rect.bottom = pic->height;

	if(FAILED(lpdds->Blt(&dest_rect, pic->lpddsImage, &source_rect, (DDBLT_WAIT | DDBLT_KEYSRC), NULL)))
	{
		PROCESS_ERROR("DD_TMap: Blt()");
		return 0;
	}

	return 1;
}


class Timer
{
public:
	void StartClock(void)
	{
		StartTime = GetTickCount();
	}

	void WaitClock(DWORD ticks)
	{
		FinishTime = StartTime + ticks;
		while(GetTickCount() < FinishTime);
	}
	
	void Sleep(DWORD ticks)
	{
		StartClock();
		WaitClock(ticks);
	}

private:
	DWORD StartTime;
	DWORD FinishTime;
};


void PROCESS_ERROR(char *error_description)
{
	ShowCursor(1);
	MessageBox(main_window_handle, error_description, "ERROR!!!", MB_OK | MB_ICONEXCLAMATION);
	DD_Shutdown();
	PostMessage(main_window_handle, WM_DESTROY, 0, 0);
}



#endif